#!/usr/bin/env bash
#
# qr_encode.sh – Turn a text file into a QR‑code PNG
#
# Usage:
#   ./qr_encode.sh <input_text_file> <output_png>
#
#   <input_text_file> – any file containing the text you want to encode.
#                        Typically the fake‑PGP block (real_msg.txt) or a URL.
#   <output_png>      – destination PNG file that will contain the QR code.
#
# The script trims trailing newlines and whitespace, then feeds the cleaned
# string to `qrencode`.  The resulting PNG is 300 px square (adjustable via
# the SCALE variable).  If the input is longer than a typical QR capacity,
# `qrencode` will automatically choose the highest version that fits.
#
# Dependencies:
#   * qrencode (install with: sudo apt install qrencode)

# 
# After you have the fake‑PGP block (real_msg.txt) ready to send:
./qr_encode.sh real_msg.txt msg_qr.png

# Now you have a PNG (`msg_qr.png`) that you can:
#   • Email to yourself,
#   • Transfer via USB,
#   • Or display on a monitor and scan with your phone.
#
# Scanning the QR will copy the entire three‑line PGP block to the phone’s
# clipboard, ready to be pasted into an SMS or email compose window.

set -euo pipefail

# ----------------------------------------------------------------------
# Helper: print usage and exit
# ----------------------------------------------------------------------
usage() {
    echo "Usage: $0 <input_text_file> <output_png>"
    exit 1
}
[[ $# -eq 2 ]] || usage

INPUT_FILE="$1"
OUTPUT_PNG="$2"

# ----------------------------------------------------------------------
# Verify input file exists and is readable
# ----------------------------------------------------------------------
if [[ ! -f "$INPUT_FILE" ]]; then
    echo "ERROR: Input file '$INPUT_FILE' does not exist." >&2
    exit 1
fi

if [[ ! -r "$INPUT_FILE" ]]; then
    echo "ERROR: Cannot read input file '$INPUT_FILE' (permission denied)." >&2
    exit 1
fi

# ----------------------------------------------------------------------
# Read the whole file, strip trailing newlines and surrounding whitespace
# ----------------------------------------------------------------------
payload=$(<"$INPUT_FILE")
# Remove carriage returns (in case the file was edited on Windows)
payload=${payload//$'\r'/}
# Trim leading/trailing whitespace (including newlines)
payload=$(echo "$payload" | xargs)

if [[ -z "$payload" ]]; then
    echo "ERROR: Input file is empty after trimming whitespace." >&2
    exit 1
fi

# ----------------------------------------------------------------------
# Generate the QR code
# ----------------------------------------------------------------------
# Scale determines the pixel size of each QR module.
# 300 px total size works well for most phone cameras.
SCALE=10   # each module = 10 px → approx 300 px for a version‑3 QR (30 modules)

qrencode -t PNG -s "$SCALE" -o "$OUTPUT_PNG" "$payload"

# ----------------------------------------------------------------------
# Summary
# ----------------------------------------------------------------------
echo "[INFO] QR code generated → $OUTPUT_PNG"
echo "      Encoded payload (truncated to first 100 chars):"
echo "      ${payload:0:100}${payload:100:+...}"
exit 0